using log4net;
using Microsoft.Azure.WebJobs;
using Microsoft.Azure.WebJobs.Extensions.Http;
using System;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.IWS.BatchService.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.IWS.CreateResponseService.Interface;

namespace VA.PPMS.IWS.Functions
{
    public static class PpmsDataStream
    {
        [FunctionName("PpmsDataStream")]
        public static async Task<HttpResponseMessage> Run([HttpTrigger(AuthorizationLevel.Anonymous, "get", Route = "PpmsDataStream/v1")]HttpRequestMessage req,
            [Inject]ICreateResponseService responseService,
            [Inject]IBatchService batchService,
            [Inject] ILog logger)
        {
            try
            {
                logger.Info("@@@@@ Start PpmsDataStream http function @@@@@");

                var content = req.GetQueryNameValuePairs().FirstOrDefault(x => string.Compare(x.Key, "id", StringComparison.OrdinalIgnoreCase) == 0).Value;

                var message = new DasMessage
                {
                    ConversationId = content,
                    TransactionId = Guid.NewGuid().ToString(),
                    SenderId = DasMessage.VaPpmsIdentifier,
                    Content = content
                };

                var batch = await responseService.GetBatchByConversationId(message.ConversationId, false);
                if (batch != null && batch.ppms_network != null)
                {
                    message.ResponseConversationId = batch.ppms_transactionid;
                    message.ReceiverId = batch.ppms_network.Name;
                }
                else
                {
                    throw new PpmsServiceException("Batch information is invalid: Unable to find associated network.");
                }

                var xmlResponse = await responseService.CreateResponse(message);

                if (!string.IsNullOrEmpty(xmlResponse))
                {
                    logger.Info($"XmlResponse: Created. Updating batch to ResponseDataRequested");
                    // If success, update batch record to reflect success at this stage
                    var updateSuccess = await batchService.UpdateBatch(message, "PPMS Data Stream request processed.", (int)ppms_batch_StatusCode.ResponseDataRequested);

                    if (updateSuccess)
                    {
                        logger.Info($"PpmsDataStream: Create response complete: {content}");
                        logger.Info("@@@@@ End PpmsDataStream http function @@@@@");
                        return req.CreateResponse(HttpStatusCode.OK, xmlResponse, "text/plain");
                    }

                    logger.Error($"PpmsDataStream: Create response failed: {content}");
                    return req.CreateResponse(HttpStatusCode.ExpectationFailed);
                }

                logger.Error($"PpmsDataStream: Create response failed: {content}");
                return req.CreateResponse(HttpStatusCode.PreconditionFailed);
            }
            catch (Exception ex)
            {
                logger.Error("An error occurred in the PpmsDataStream Function API: ", ex);
                return req.CreateErrorResponse(HttpStatusCode.InternalServerError, "An error occurred in the PpmsDataStream Function API: ", ex);
            }
        }
    }
}